package com.tangguo.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.mauth.MobileTokenHelper;
import com.tangguo.domain.BbsMoment;
import com.tangguo.domain.BbsMomentAttachment;
import com.tangguo.domain.BbsMomentVoteOption;
import com.tangguo.domain.bo.CreateMomentBO;
import com.tangguo.domain.vo.BbsMomentListVO;
import com.tangguo.domain.vo.BbsUserMomentCountVO;
import com.tangguo.enums.EnableStatus;
import com.tangguo.enums.MomentType;
import com.tangguo.enums.VoteOptionType;
import com.tangguo.mapper.BbsMomentMapper;
import com.tangguo.service.IBbsMomentAttachmentService;
import com.tangguo.service.IBbsMomentService;
import com.tangguo.service.IBbsMomentVoteOptionService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 动态Service业务层处理
 *
 * @author ruoyi
 * @date 2025-09-01
 */
@Slf4j
@Service
public class BbsMomentServiceImpl extends ServiceImpl<BbsMomentMapper, BbsMoment> implements IBbsMomentService {

    @Resource
    private BbsMomentMapper bbsMomentMapper;

    @Resource
    private IBbsMomentAttachmentService attachmentsService;

    @Resource
    private IBbsMomentVoteOptionService voteOptionService;


    /**
     * 查询动态列表
     *
     * @param moment 动态
     * @return 动态
     */
    @Override
    public List<BbsMoment> selectBbsMomentList(BbsMoment moment) {
	    return this.baseMapper.selectBbsMomentList(moment);
    }


    /**
     * 删除动态
     *
     * @param id 动态Id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteMoment(Long id) {
        this.removeById(id);
    }


    /**
     * 查询用户动态统计
     *
     * @return 动态统计
     */
    @Override
    public BbsUserMomentCountVO getUserMomentCount(String userName) {
        return this.baseMapper.selectUserMomentCount(userName);
    }


    /**
     * 查询动态列表
     *
     * @param userName 用户名
     * @param content  动态内容
     * @return 动态列表
     */
    @Override
    public List<BbsMomentListVO> getUserMoments(String userName, String content) {
        List<BbsMomentListVO> dbMoments = this.baseMapper.selectUserMoments(userName, content);
        this.fillMoment(dbMoments);
        return dbMoments;
    }


    /**
     * 查询动态列表
     *
     * @param userName 用户名
     * @param content  动态内容
     * @return 动态列表
     */
    @Override
    public List<BbsMomentListVO> getMoments(String userName, String content) {
        List<BbsMomentListVO> dbMoments = this.baseMapper.selectMoments(userName, content);
        this.fillMoment(dbMoments);
        return dbMoments;
    }


    /**
     * 创建动态
     *
     * @param bo 动态
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void userCreateMoment(CreateMomentBO bo) {
        // 添加动态
        String username = MobileTokenHelper.getUsername();
        BbsMoment newMoment = this.buildMoment(bo);
        newMoment.setUserName(username);
        this.save(newMoment);

        // 添加动态附件
        MomentType momentType = MomentType.getMomentType(bo.getType());
        if (MomentType.IMAGE == momentType || MomentType.VIDEO == momentType) {
            this.attachmentsService.addAttachments(newMoment, bo.getAttachments());
        }

        // 添加动态投票选项
        if (EnableStatus.QY.getStatus() == bo.getIsEnableVote()) {
            List<CreateMomentBO.VoteOption> voteOptions = bo.getVoteOptions();
            this.voteOptionService.addVoteOptions(newMoment, voteOptions);
        }
    }


    /**
     * 删除动态
     *
     * @param momentId 动态Id
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void userDeleteMoment(Long momentId) {
        String username = MobileTokenHelper.getUsername();
        BbsMoment dbMoment = this.getOne(
            Wrappers.lambdaQuery(BbsMoment.class)
                .select(BbsMoment::getId).eq(BbsMoment::getUserName, username).eq(BbsMoment::getId, momentId)
        );
        if (Objects.isNull(dbMoment)) {
            throw new ServiceException("删除失败，未查询到当前动态数据。");
        }

        this.removeById(dbMoment.getId());
    }


    /**
     * 构建动态实体
     *
     * @param bo 动态
     * @return 动态实体
     */
    private BbsMoment buildMoment(CreateMomentBO bo) {
        // 动态类型
        MomentType momentType = MomentType.getMomentType(bo.getType());
        if (Objects.isNull(momentType)) {
            throw new ServiceException("发布失败，动态类型错误。");
        }

        // 动态附件
        BbsMoment newMoment = new BbsMoment();
        List<CreateMomentBO.Attachment> attachments = bo.getAttachments();
        String linkUrl = bo.getLinkUrl();
        if (MomentType.IMAGE == momentType || MomentType.VIDEO == momentType) {
            if (CollUtil.isEmpty(attachments)) {
                throw new ServiceException("发布失败，动态附件不能为空。");
            }
        }
        else if (MomentType.URL == momentType) {
            if (StrUtil.isBlank(linkUrl)) {
                throw new ServiceException("发布失败，动态链接不能为空。");
            } else {
                newMoment.setLinkUrl(linkUrl);
            }
        }

        newMoment.setType(momentType.name());
        newMoment.setContent(bo.getContent());
        newMoment.setIsEnableComment(bo.getIsEnableComment());
        newMoment.setIsEnableFeaturedComment(bo.getIsEnableFeaturedComment());

        // 动态主题
        List<String> topicNames = bo.getTopicNames();
        List<String> topicIds = bo.getTopicIds();
        if (CollUtil.isNotEmpty(topicNames) && CollUtil.isNotEmpty(topicIds)) {
            newMoment.setTopicIds(String.join(",", topicIds));
            newMoment.setTopicNames(String.join(",", topicNames));
        }

        // 动态投票
        Integer isEnableVote = bo.getIsEnableVote();
        newMoment.setIsEnableVote(isEnableVote);
        if (EnableStatus.QY.getStatus() == isEnableVote) {
            newMoment.setVoteTitle(bo.getVoteTitle());
            VoteOptionType optionType = VoteOptionType.getVoteOptionType(bo.getVoteOptionType());
            if (Objects.isNull(optionType)) {
                throw new ServiceException("发布失败，动态投票选项类型错误。");
            }

            newMoment.setVoteOptionType(optionType.name());
            List<CreateMomentBO.VoteOption> options = bo.getVoteOptions();
            if (CollUtil.isEmpty(options)) {
                throw new ServiceException("发布失败，动态投票选项不能为空。");
            }

            for (CreateMomentBO.VoteOption option : options) {
                long optionCount = options.stream()
                    .filter(o -> o.getName().equals(option.getName())).count();
                if (optionCount > 1) {
                    throw new ServiceException("发布失败，动态投票选项重复。");
                }
            }
        }
        return newMoment;
    }


    /**
     * 填充动态附件和投票选项
     *
     * @param moments 动态
     */
    private void fillMoment(List<BbsMomentListVO> moments) {
        List<Long> momentIds = moments.stream().map(BbsMomentListVO::getId).collect(Collectors.toList());
        if (CollUtil.isEmpty(momentIds)) {
            return;
        }

        try {
            // 查询动态附件
            CompletableFuture<Map<Long, List<BbsMomentAttachment>>> af = CompletableFuture.supplyAsync(() -> {
                List<BbsMomentAttachment> atts = this.attachmentsService.getAttachments(momentIds);
                return atts.stream().collect(Collectors.groupingBy(BbsMomentAttachment::getMomentId));
            });

            // 查询动态投票选项
            CompletableFuture<Map<Long, List<BbsMomentVoteOption>>> of = CompletableFuture.supplyAsync(() -> {
                List<BbsMomentVoteOption> opts = this.voteOptionService.getVoteOptions(momentIds);
                return opts.stream().collect(Collectors.groupingBy(BbsMomentVoteOption::getMomentId));
            });

            // 等待全部任务处理完毕
            CompletableFuture.allOf(af, of).join();
            Map<Long, List<BbsMomentAttachment>> attsMap = af.join();
            Map<Long, List<BbsMomentVoteOption>> optsMap = of.join();

            // 填充动态
            for (BbsMomentListVO m : moments) {
                // 填充动态附件
                List<BbsMomentAttachment> as = attsMap.get(m.getId());
                if (CollUtil.isNotEmpty(as)) {
                    List<BbsMomentListVO.Attachment> avs = new ArrayList<>(as.size());
                    for (BbsMomentAttachment a : as) {
                        BbsMomentListVO.Attachment av = new BbsMomentListVO.Attachment();
                        av.setName(a.getName());
                        av.setUrl(a.getUrl());
                        avs.add(av);
                    }
                    m.setAttachments(avs);
                }

                // 填充动态投票选项
                List<BbsMomentVoteOption> os = optsMap.get(m.getId());
                if (CollUtil.isNotEmpty(os)) {
                    List<BbsMomentListVO.VoteOption> ovs = new ArrayList<>(os.size());
                    for (BbsMomentVoteOption o : os) {
                        BbsMomentListVO.VoteOption ov = new BbsMomentListVO.VoteOption();
                        ov.setType(o.getType());
                        ov.setName(o.getName());
                        ov.setCode(o.getCode());
                        ov.setImageUrl(o.getImageUrl());
                        ov.setSort(o.getSort());
                        ov.setVoteCount(o.getVoteCount());
                        ovs.add(ov);
                    }
                    m.setVoteOptions(ovs);
                }
            }

        } catch (Exception e) {
            log.error("=> 填充动态附件和投票选项失败：", e);
        }
    }

}
