package com.tangguo.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.mauth.MobileTokenHelper;
import com.tangguo.domain.BbsMoment;
import com.tangguo.domain.bo.CreateMomentBO;
import com.tangguo.domain.vo.BbsUserMomentCountVO;
import com.tangguo.enums.EnableStatus;
import com.tangguo.mapper.BbsMomentMapper;
import com.tangguo.service.IBbsMomentAttachmentService;
import com.tangguo.service.IBbsMomentService;
import com.tangguo.service.IBbsMomentVoteOptionService;
import org.apache.ibatis.parsing.TokenHandler;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/**
 * 动态Service业务层处理
 *
 * @author ruoyi
 * @date 2025-09-01
 */
@Service
public class BbsMomentServiceImpl extends ServiceImpl<BbsMomentMapper, BbsMoment> implements IBbsMomentService {

    @Resource
    private BbsMomentMapper bbsMomentMapper;

    @Resource
    private IBbsMomentAttachmentService attachmentsService;

    @Resource
    private IBbsMomentVoteOptionService voteOptionService;


    /**
     * 查询动态列表
     *
     * @param moment 动态
     * @return 动态
     */
    @Override
    public List<BbsMoment> selectBbsMomentList(BbsMoment moment) {
	    return this.baseMapper.selectBbsMomentList(moment);
    }


    /**
     * 查询用户动态相关统计数据
     *
     * @return 动态统计数据
     */
    @Override
    public BbsUserMomentCountVO getUserMomentCount() {
        String username = MobileTokenHelper.getUsername();
        return this.baseMapper.selectUserMomentCount(username);
    }


    /**
     * 创建动态
     *
     * @param bo 动态
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void createMoment(CreateMomentBO bo) {
        // 添加动态
        String username = MobileTokenHelper.getUsername();
        BbsMoment newMoment = this.buildBbsMoment(bo);
        newMoment.setUserName(username);
        this.save(newMoment);

        // 添加动态附件
        List<CreateMomentBO.Attachment> attachments = bo.getAttachments();
        this.attachmentsService.addAttachments(username, newMoment.getId(), attachments);

        // 添加动态投票选项
        if (EnableStatus.QY.getStatus() == bo.getIsEnableVote()) {
            List<CreateMomentBO.VoteOption> voteOptions = bo.getVoteOptions();
            this.voteOptionService.addVoteOptions(newMoment.getId(), voteOptions);
        }
    }


    /**
     * 构建动态实体
     *
     * @param bo 动态
     * @return 动态实体
     */
    private BbsMoment buildBbsMoment(CreateMomentBO bo) {
        BbsMoment newMoment = new BbsMoment();
        newMoment.setContent(bo.getContent());
        newMoment.setType(bo.getType());
        newMoment.setLinkUrl(bo.getLinkUrl());
        newMoment.setIsEnableComment(bo.getIsEnableComment());
        newMoment.setIsEnableFeaturedComment(bo.getIsEnableFeaturedComment());

        // 动态主题
        List<String> topicNames = bo.getTopicNames();
        if (CollUtil.isNotEmpty(topicNames)) {
            newMoment.setTopicNames(String.join(",", topicNames));
        }
        List<String> topicIds = bo.getTopicIds();
        if (CollUtil.isNotEmpty(topicIds)) {
            newMoment.setTopicIds(String.join(",", topicIds));
        }

        // 动态附件
        List<String> attachmentUrls = bo.getAttachmentUrls();
        if (CollUtil.isNotEmpty(attachmentUrls)) {
            newMoment.setAttachmentUrls(String.join(",", attachmentUrls));
        }

        // 动态投票
        Integer isEnableVote = bo.getIsEnableVote();
        newMoment.setIsEnableVote(isEnableVote);
        if (EnableStatus.QY.getStatus() == isEnableVote) {
            newMoment.setVoteTitle(bo.getVoteTitle());
            newMoment.setVoteOptionType(bo.getVoteOptionType());
            List<CreateMomentBO.VoteOption> voteOptions = bo.getVoteOptions();
            for (int i = 0; i < voteOptions.size(); i++) {
                int index = i + 1;
                CreateMomentBO.VoteOption option = voteOptions.get(i);
                option.setType(bo.getVoteOptionType());
                option.setCode(String.valueOf(index));
                option.setSort(index);
            }
            newMoment.setVoteOptions(JSONUtil.toJsonStr(voteOptions));
        }
        return newMoment;
    }

}
