package com.tangguo.controller;

import com.tangguo.common.annotation.Log;
import com.tangguo.common.core.controller.BaseController;
import com.tangguo.common.core.domain.AjaxResult;
import com.tangguo.common.enums.BusinessType;
import com.tangguo.domain.BbsPointsSetting;
import com.tangguo.enums.PointsSettingType;
import com.tangguo.service.IBbsPointsSettingService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 积分规则配置Controller
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Api(tags = "积分规则配置管理")
@RestController
@RequestMapping("/bbs/points/setting")
public class BbsPointsSettingController extends BaseController {

	@Resource
	private IBbsPointsSettingService bbsPointsSettingService;


	/**
	 * 查询积分规则配置列表
	 */
	@ApiOperation("查询积分规则配置列表")
	@PreAuthorize("@ss.hasPermi('points:points:list')")
	@GetMapping("/details")
	public AjaxResult details() {
		Map<String, List<BbsPointsSetting>> details = this.bbsPointsSettingService.getSettingDetails();
		return AjaxResult.success(details);
	}


	/**
	 * 获取积分规则配置详细信息
	 */
	@ApiOperation("获取积分规则配置详细信息")
	@PreAuthorize("@ss.hasPermi('points:points:query')")
	@GetMapping(value = "/{id}")
	public AjaxResult getInfo(@PathVariable("id") Long id) {
		return success(bbsPointsSettingService.getById(id));
	}


	/**
	 * 新增积分规则配置
	 */
	@ApiOperation("新增积分规则配置")
	@PreAuthorize("@ss.hasPermi('points:points:add')")
	@Log(title = "积分规则配置", businessType = BusinessType.INSERT)
	@PostMapping
	public AjaxResult add(@RequestBody BbsPointsSetting setting) {
		this.bbsPointsSettingService.addSetting(setting);
		return AjaxResult.success();
	}


	/**
	 * 修改积分规则配置
	 */
	@ApiOperation("修改积分规则配置")
	@PreAuthorize("@ss.hasPermi('points:points:edit')")
	@Log(title = "积分规则配置", businessType = BusinessType.UPDATE)
	@PutMapping
	public AjaxResult edit(@RequestBody BbsPointsSetting setting) {
		this.bbsPointsSettingService.editSetting(setting);
		return AjaxResult.success();
	}


	/**
	 * 修改积分规则配置
	 */
	@ApiOperation("修改积分规则配置")
	@PreAuthorize("@ss.hasPermi('points:points:edit')")
	@Log(title = "积分规则配置", businessType = BusinessType.UPDATE)
	@PutMapping("/batch")
	public AjaxResult batchEdit(@RequestBody Map<String, List<BbsPointsSetting>> pointMap) {
		List<BbsPointsSetting> adds = pointMap.get(PointsSettingType.ADD.name());
		List<BbsPointsSetting> decs = pointMap.get(PointsSettingType.DEC.name());
		List<BbsPointsSetting> upds = new ArrayList<>(15);
		upds.addAll(adds);
		upds.addAll(decs);
		this.bbsPointsSettingService.batchEditSetting(upds);
		return AjaxResult.success();
	}



	/**
	 * 删除积分规则配置
	 */
	@ApiOperation("删除积分规则配置")
	@PreAuthorize("@ss.hasPermi('points:points:remove')")
	@Log(title = "积分规则配置", businessType = BusinessType.DELETE)
	@DeleteMapping("/{id}")
	public AjaxResult remove(@PathVariable Long id) {
		this.bbsPointsSettingService.deleteSetting(id);
		return AjaxResult.success();
	}

}
