package com.tangguo.service.impl;

import cn.hutool.core.util.IdUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.domain.BbsPointsSetting;
import com.tangguo.enums.PointsSettingType;
import com.tangguo.mapper.BbsPointsSettingMapper;
import com.tangguo.service.IBbsPointsSettingService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 积分规则配置Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Service
public class BbsPointsSettingServiceImpl extends ServiceImpl<BbsPointsSettingMapper, BbsPointsSetting> implements IBbsPointsSettingService {

    @Resource
    private BbsPointsSettingMapper bbsPointsSettingMapper;


    /**
     * 查询积分规则配置列表
     *
     * @param bbsPointsSetting 积分规则配置
     * @return 积分规则配置
     */
    @Override
    public List<BbsPointsSetting> selectBbsPointsSettingList(BbsPointsSetting bbsPointsSetting) {
	    return this.list();
    }


    /**
     * 查询积分规则配置明细
     *
     * @return 配置明细
     */
    @Override
    public Map<String, List<BbsPointsSetting>> getSettingDetails() {
        List<BbsPointsSetting> settings = this.list();

        List<BbsPointsSetting> addSetting = settings.stream()
            .filter(s -> PointsSettingType.ADD.name().equals(s.getType())).collect(Collectors.toList());
        List<BbsPointsSetting> decSetting = settings.stream()
            .filter(s -> PointsSettingType.DEC.name().equals(s.getType())).collect(Collectors.toList());

        Map<String, List<BbsPointsSetting>> settingMap = new LinkedHashMap<>(2);
        settingMap.put(PointsSettingType.ADD.name(), addSetting);
        settingMap.put(PointsSettingType.DEC.name(), decSetting);
        return settingMap;
    }


    /**
     * 添加积分配置
     *
     * @param setting 积分配置
     */
    @Override
    public void addSetting(BbsPointsSetting setting) {
        long nameCount = this.count(
            Wrappers.lambdaQuery(BbsPointsSetting.class).eq(BbsPointsSetting::getOperateName, setting.getOperateName())
        );
        if (nameCount > 0) {
            throw new ServiceException("新增失败，操作行为名称重复。");
        }

        BbsPointsSetting addSetting = new BbsPointsSetting();
        addSetting.setType(setting.getType());
        addSetting.setAgentName(setting.getAgentName());
        addSetting.setOperateName(setting.getOperateName());
        addSetting.setOperateCode(IdUtil.fastSimpleUUID());
        addSetting.setOperatePoints(setting.getOperatePoints());
        addSetting.setLimitCount(setting.getLimitCount());
        addSetting.setEnableStatus(setting.getEnableStatus());
        this.save(addSetting);
    }


    /**
     * 修改积分配置
     *
     * @param setting 积分配置
     */
    @Override
    public void editSetting(BbsPointsSetting setting) {
        BbsPointsSetting dbSetting = this.getById(setting.getId());
        if (Objects.isNull(dbSetting)) {
            throw new ServiceException("修改失败，未查询到当前积分配置数据。");
        }

        if (!dbSetting.getOperateName().equals(setting.getOperateName())) {
            long nameCount = this.count(
                 Wrappers.lambdaQuery(BbsPointsSetting.class).eq(BbsPointsSetting::getOperateName, setting.getOperateName())
            );
            if (nameCount > 0) {
                throw new ServiceException("修改失败，操作行为名称重复。");
            }
        }

        BbsPointsSetting updSetting = new BbsPointsSetting();
        updSetting.setId(dbSetting.getId());
        updSetting.setOperateName(setting.getOperateName());
        updSetting.setOperatePoints(setting.getOperatePoints());
        updSetting.setLimitCount(setting.getLimitCount());
        updSetting.setEnableStatus(setting.getEnableStatus());
        this.updateById(updSetting);
    }


    /**
     * 删除积分配置
     *
     * @param settingId 配置Id
     */
    @Override
    public void deleteSetting(Long settingId) {
        this.removeById(settingId);
    }


    /**
     * 修改配置状态
     *
     * @param setting 积分配置
     */
    @Override
    public void setSettingStatus(BbsPointsSetting setting) {
        BbsPointsSetting dbSetting = this.getById(setting.getId());
        if (Objects.isNull(dbSetting)) {
            throw new ServiceException("修改失败，未查询到当前积分配置数据。");
        }

        if (!dbSetting.getEnableStatus().equals(setting.getEnableStatus())) {
            BbsPointsSetting updSetting = new BbsPointsSetting();
            updSetting.setId(dbSetting.getId());
            updSetting.setEnableStatus(setting.getEnableStatus());
            this.updateById(updSetting);
        }
    }

}
