package com.tangguo.controller;

import com.tangguo.common.annotation.Log;
import com.tangguo.common.core.controller.BaseController;
import com.tangguo.common.core.domain.AjaxResult;
import com.tangguo.common.core.page.TableDataInfo;
import com.tangguo.common.enums.BusinessType;
import com.tangguo.common.utils.poi.ExcelUtil;
import com.tangguo.domain.BbsUserPoints;
import com.tangguo.service.IBbsUserPointsService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 用户积分Controller
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Api(tags = "用户积分管理")
@RestController
@RequestMapping("/points/points")
public class BbsUserPointsController extends BaseController {

	@Resource
	private IBbsUserPointsService bbsUserPointsService;


	/**
	 * 查询用户积分列表
	 */
	@ApiOperation("查询用户积分列表")
	@PreAuthorize("@ss.hasPermi('points:points:list')")
	@GetMapping("/list")
	public TableDataInfo list(BbsUserPoints bbsUserPoints) {
		startPage();
		List<BbsUserPoints> list = bbsUserPointsService.selectBbsUserPointsList(bbsUserPoints);
		return getDataTable(list);
	}


	/**
	 * 导出用户积分列表
	 */
	@ApiOperation("导出用户积分列表")
	@PreAuthorize("@ss.hasPermi('points:points:export')")
	@Log(title = "用户积分", businessType = BusinessType.EXPORT)
	@PostMapping("/export")
	public void export(HttpServletResponse response, BbsUserPoints bbsUserPoints) {
		List<BbsUserPoints> list = bbsUserPointsService.selectBbsUserPointsList(bbsUserPoints);
		ExcelUtil<BbsUserPoints> util = new ExcelUtil<BbsUserPoints>(BbsUserPoints.class);
		util.exportExcel(response, list, "用户积分数据");
	}


	/**
	 * 获取用户积分详细信息
	 */
	@ApiOperation("获取用户积分详细信息")
	@PreAuthorize("@ss.hasPermi('points:points:query')")
	@GetMapping(value = "/{id}")
	public AjaxResult getInfo(@PathVariable("id") Long id) {
		return success(bbsUserPointsService.getById(id));
	}


	/**
	 * 新增用户积分
	 */
	@ApiOperation("新增用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:add')")
	@Log(title = "用户积分", businessType = BusinessType.INSERT)
	@PostMapping
	public AjaxResult add(@RequestBody BbsUserPoints bbsUserPoints) {
		return toAjax(bbsUserPointsService.save(bbsUserPoints));
	}


	/**
	 * 修改用户积分
	 */
	@ApiOperation("修改用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:edit')")
	@Log(title = "用户积分", businessType = BusinessType.UPDATE)
	@PutMapping
	public AjaxResult edit(@RequestBody BbsUserPoints bbsUserPoints) {
		return toAjax(bbsUserPointsService.updateById(bbsUserPoints));
	}


	/**
	 * 删除用户积分
	 */
	@ApiOperation("删除用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:remove')")
	@Log(title = "用户积分", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
	public AjaxResult remove(@PathVariable Long[] ids) {
		return toAjax(bbsUserPointsService.removeByIds(Arrays.asList(ids)));
	}

}
