package com.tangguo.controller;

import com.tangguo.common.annotation.Log;
import com.tangguo.common.annotation.RepeatSubmit;
import com.tangguo.common.core.controller.BaseController;
import com.tangguo.common.core.domain.AjaxResult;
import com.tangguo.common.core.page.TableDataInfo;
import com.tangguo.common.enums.BusinessType;
import com.tangguo.common.utils.poi.ExcelUtil;
import com.tangguo.domain.BbsUserPoints;
import com.tangguo.domain.BbsUserPointsDetail;
import com.tangguo.domain.bo.BatchUserPointsBO;
import com.tangguo.service.IBbsUserPointsDetailService;
import com.tangguo.service.IBbsUserPointsService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 用户积分Controller
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Api(tags = "用户积分管理")
@RestController
@RequestMapping("/bbs/points/user/points")
public class BbsUserPointsController extends BaseController {

	@Resource
	private IBbsUserPointsService bbsUserPointsService;

	@Resource
	private IBbsUserPointsDetailService pointsDetailService;


	/**
	 * 查询用户积分列表
	 */
	@ApiOperation("查询用户积分列表")
	@GetMapping("/list")
	public TableDataInfo list(BbsUserPoints points) {
		startPage();
		List<BbsUserPoints> list = bbsUserPointsService.selectBbsUserPointsList(points);
		return getDataTable(list);
	}


	/**
	 * 查询用户积分
	 */
	@ApiOperation("查询用户积分")
	@GetMapping("/{userName}")
	public AjaxResult list(@PathVariable String userName) {
		BbsUserPoints dbPoints = this.bbsUserPointsService.selectBbsUserPoints(userName);
		return AjaxResult.success(dbPoints);
	}


	/**
	 * 查询用户积分明细
	 */
	@ApiOperation("查询用户积分明细")
	@PreAuthorize("@ss.hasPermi('points:points:list')")
	@GetMapping("/details")
	public TableDataInfo details(String userName) {
		startPage();
		List<BbsUserPointsDetail> details = this.pointsDetailService.getUserPointsDetail(userName);
		return getDataTable(details);
	}


	/**
	 * 导出用户积分列表
	 */
	@ApiOperation("导出用户积分列表")
	@PreAuthorize("@ss.hasPermi('points:points:export')")
	@Log(title = "用户积分", businessType = BusinessType.EXPORT)
	@PostMapping("/export")
	public void export(HttpServletResponse response, BbsUserPoints points) {
		List<BbsUserPoints> list = bbsUserPointsService.selectBbsUserPointsList(points);
		ExcelUtil<BbsUserPoints> util = new ExcelUtil<>(BbsUserPoints.class);
		util.exportExcel(response, list, "用户积分数据");
	}


	/**
	 * 增加用户积分
	 */
	@RepeatSubmit(interval = 1000)
	@ApiOperation("新增用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:incr')")
	@Log(title = "用户积分", businessType = BusinessType.UPDATE)
	@PutMapping("/incr")
	public AjaxResult incrUserPoints(@RequestBody BbsUserPoints points) {
		this.bbsUserPointsService.addUserPoints(points);
		return AjaxResult.success();
	}


	/**
	 * 扣减用户积分
	 */
	@RepeatSubmit(interval = 1000)
	@ApiOperation("扣减用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:decr')")
	@Log(title = "用户积分", businessType = BusinessType.UPDATE)
	@PutMapping("/decr")
	public AjaxResult decrUserPoints(@RequestBody BbsUserPoints points) {
		this.bbsUserPointsService.deleteUserPoints(points);
		return AjaxResult.success();
	}


	/**
	 * 增加用户积分
	 */
	@RepeatSubmit(interval = 1000)
	@ApiOperation("新增用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:incr')")
	@Log(title = "用户积分", businessType = BusinessType.UPDATE)
	@PutMapping("/batch/incr")
	public AjaxResult batchIncrUserPoints(@RequestBody BatchUserPointsBO bo) {
		this.bbsUserPointsService.batchAddUserPoints(bo);
		return AjaxResult.success();
	}


	/**
	 * 扣减用户积分
	 */
	@RepeatSubmit(interval = 1000)
	@ApiOperation("扣减用户积分")
	@PreAuthorize("@ss.hasPermi('points:points:decr')")
	@Log(title = "用户积分", businessType = BusinessType.UPDATE)
	@PutMapping("/batch/decr")
	public AjaxResult batchDecrUserPoints(@RequestBody BatchUserPointsBO bo) {
		this.bbsUserPointsService.batchDeleteUserPoints(bo);
		return AjaxResult.success();
	}

}
