package com.tangguo.service.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.domain.PointsDetail;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.utils.SecurityUtils;
import com.tangguo.domain.BbsGrade;
import com.tangguo.domain.BbsUserPoints;
import com.tangguo.domain.BbsUserPointsDetail;
import com.tangguo.domain.bo.BatchUserPointsBO;
import com.tangguo.mapper.BbsUserPointsMapper;
import com.tangguo.service.IBbsGradeService;
import com.tangguo.service.IBbsUserPointsDetailService;
import com.tangguo.service.IBbsUserPointsService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import javax.annotation.Resource;
import java.util.*;

/**
 * 用户积分Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Slf4j
@Service
public class BbsUserPointsServiceImpl extends ServiceImpl<BbsUserPointsMapper, BbsUserPoints> implements IBbsUserPointsService {

    @Resource
    private BbsUserPointsMapper bbsUserPointsMapper;

    @Resource
    private BbsUserPointsServiceImpl pointsServiceImpl;

    @Resource
    private IBbsUserPointsDetailService pointsDetailService;

    @Resource
    private IBbsGradeService gradeService;

	@Resource
	private TransactionTemplate transactionTemplate;



    /**
     * 查询用户积分列表
     *
     * @param points 用户积分
     * @return 用户积分
     */
    @Override
    public List<BbsUserPoints> selectBbsUserPointsList(BbsUserPoints points) {
        return this.baseMapper.selectBbsUserPointsList(points);
    }


    /**
     * 查询用户积分
     *
     * @param userName 用户名
     * @return 用户积分集合
     */
    @Override
    public BbsUserPoints selectBbsUserPoints(String userName) {
        return this.baseMapper.selectBbsUserPoints(userName);
    }


    /**
     * 查询用户积分
     *
     * @param userName 用户名
     * @return 积分
     */
    @Override
    public BbsUserPoints getUserPoints(String userName) {
        return this.getOne(
            Wrappers.lambdaQuery(BbsUserPoints.class).eq(BbsUserPoints::getUserName, userName)
        );
    }


    /**
     * 查询用户积分
     *
     * @param userName 用户名
     * @return 积分
     */
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    @Override
    public BbsUserPoints getAndInitUserPoints(String userName) {
        BbsUserPoints dbUserPoints = this.getUserPoints(userName);
        if (Objects.isNull(dbUserPoints)) {
            int dbUserCount = this.baseMapper.selectSysUserCount(userName);
            if (dbUserCount > 0) {
                // 初始积分
                dbUserPoints = new BbsUserPoints();
                dbUserPoints.setUserName(userName);
                dbUserPoints.setAccumulatedPoints(0);
                dbUserPoints.setCurrentPoints(0);
                // 初始等级
                BbsGrade dbGrade = this.gradeService.getInitialGrade();
                if (Objects.nonNull(dbGrade)) {
                    dbUserPoints.setGradeCode(dbGrade.getCode());
                    dbUserPoints.setGradeName(dbGrade.getName());
                    dbUserPoints.setLastUpgradeTime(new Date());
                }
                this.save(dbUserPoints);
            } else {
                throw new ServiceException("查询用户积分失败，未查询到当前用户数据。");
            }
        }
        return dbUserPoints;
    }


    /**
     * 增加用户积分
     *
     * @param points 积分
     */
    @Override
    public void addUserPoints(BbsUserPoints points) {
        String userName = points.getUserName();
        PointsDetail detail = new PointsDetail();
        detail.setUserName(userName);
        detail.setDetailPoints(points.getIncrOrDecrPoints());
        detail.setDetailName("后台手动增加");
	    String remarks = points.getRemarks();
	    if (StrUtil.isBlank(remarks)) {
		    remarks = String.format("管理员【%s】后台手动增加用户【%s】积分", SecurityUtils.getUsername(), userName);
	    }
	    detail.setDescription(remarks);
        this.incrUserPoints(detail);
    }


    /**
     * 扣减用户积分
     *
     * @param points 积分
     */
    @Override
    public void deleteUserPoints(BbsUserPoints points) {
        String userName = points.getUserName();
        PointsDetail detail = new PointsDetail();
        detail.setUserName(userName);
        detail.setDetailPoints(points.getIncrOrDecrPoints());
        detail.setDetailName("后台手动扣减");
	    String remarks = points.getRemarks();
		if (StrUtil.isBlank(remarks)) {
			remarks = String.format("管理员【%s】后台手动扣减用户【%s】积分", SecurityUtils.getUsername(), userName);
		}
        detail.setDescription(remarks);
        this.decrUserPoints(detail);
    }


	/**
	 * 增加用户积分
	 *
	 * @param bo 积分
	 */
	@Override
	public List<Map<String, String>> batchAddUserPoints(BatchUserPointsBO bo) {
		List<Map<String, String>> results = new ArrayList<>(10);
		for (BatchUserPointsBO.User user : bo.getUsers()) {
			String userName = user.getUserName();
			try {
				PointsDetail detail = new PointsDetail();
				detail.setUserName(userName);
				detail.setDetailPoints(bo.getPoints());
				detail.setDetailName("后台手动增加");
				String remarks = bo.getRemarks();
				if (StrUtil.isBlank(remarks)) {
					remarks = String.format("管理员【%s】后台手动增加用户【%s】积分", SecurityUtils.getUsername(), userName);
				}
				detail.setDescription(remarks);
				this.incrUserPoints(detail);

			} catch (Exception e) {
				log.error("=> 用户：{} 增加积分：{} 失败：", userName, bo.getPoints(), e);
				Map<String, String> resultItem = new HashMap<>(2);
				resultItem.put("nickName", user.getNickName());
				resultItem.put("userName", userName);
				resultItem.put("errorMessage", e.getMessage());
				results.add(resultItem);
			}
		}
		return results;
	}


	/**
	 * 扣减用户积分
	 *
	 * @param bo 积分
	 */
	@Override
	public List<Map<String, String>> batchDeleteUserPoints(BatchUserPointsBO bo) {
		List<Map<String, String>> results = new ArrayList<>(10);
		for (BatchUserPointsBO.User user : bo.getUsers()) {
			String userName = user.getUserName();
			try {
				PointsDetail detail = new PointsDetail();
				detail.setUserName(userName);
				detail.setDetailPoints(bo.getPoints());
				detail.setDetailName("后台手动增加");
				String remarks = bo.getRemarks();
				if (StrUtil.isBlank(remarks)) {
					remarks = String.format("管理员【%s】后台手动扣减用户【%s】积分", SecurityUtils.getUsername(), userName);
				}
				detail.setDescription(remarks);
				this.decrUserPoints(detail);

			} catch (Exception e) {
				log.error("=> 用户：{} 扣减积分：{} 失败：", userName, bo.getPoints(), e);
				Map<String, String> resultItem = new HashMap<>(2);
				resultItem.put("nickName", user.getNickName());
				resultItem.put("userName", userName);
				resultItem.put("errorMessage", e.getMessage());
				results.add(resultItem);
			}
		}
		return results;
	}


	/**
     * 增加用户积分
     *
     * @param detail   积分明细
     */
    @Override
    public void incrUserPoints(PointsDetail detail) {
        synchronized (detail.getUserName().intern()) {
            this.pointsServiceImpl.privIncrUserPoints(detail);
        }
    }


    /**
     * 扣减用户积分
     *
     * @param detail   积分明细
     */
    @Override
    public void decrUserPoints(PointsDetail detail) {
        synchronized (detail.getUserName().intern()) {
            this.pointsServiceImpl.privDecrUserPoints(detail);
        }
    }


    /**
     * 增加用户积分
     *
     * @param detail   积分明细
     */
    public void privIncrUserPoints(PointsDetail detail) {
        int points = detail.getDetailPoints();
        if (points < 1) {
            throw new ServiceException("增加用户积分失败，增加的积分分值不能小于0。");
        }

		this.transactionTemplate.executeWithoutResult(status -> {
			try {
				// 更新用户积分
				String userName = detail.getUserName();
				BbsUserPoints dbUserPoints = this.pointsServiceImpl.getAndInitUserPoints(userName);
				int beforeCurrentPoints = dbUserPoints.getCurrentPoints();
				int afterCurrentPoints = beforeCurrentPoints + points;
				int accumulatedPoints = dbUserPoints.getAccumulatedPoints() + points;
				dbUserPoints.setCurrentPoints(afterCurrentPoints);
				dbUserPoints.setAccumulatedPoints(accumulatedPoints);

				// 更新用户等级
				BbsGrade dbGrade = this.gradeService.getGradeByPoints(accumulatedPoints);
				if (Objects.nonNull(dbGrade) && !dbGrade.getCode().equals(dbUserPoints.getGradeCode())) {
					dbUserPoints.setGradeName(dbGrade.getName());
					dbUserPoints.setGradeCode(dbGrade.getCode());
					dbUserPoints.setLastUpgradeTime(new Date());
				}
				this.updateById(dbUserPoints);

				// 添加用户积分明细
				BbsUserPointsDetail newDetail = new BbsUserPointsDetail();
				newDetail.setUserName(userName);
				newDetail.setDetailName(detail.getDetailName());
				newDetail.setDetailCode(detail.getDetailCode());
				newDetail.setDetailPoints(points);
				newDetail.setBeforePoints(beforeCurrentPoints);
				newDetail.setAfterPoints(afterCurrentPoints);
				newDetail.setDescription(detail.getDescription());
				this.pointsDetailService.save(newDetail);

			} catch (Exception e) {
				log.error("=> 处理增加用户积分失败：", e);
				status.setRollbackOnly();
				throw new ServiceException(e.getMessage());
			}
		});
    }


    /**
     * 扣减用户积分
     *
     * @param detail   积分明细
     */
    public void privDecrUserPoints(PointsDetail detail) {
        int points = detail.getDetailPoints();
        if (points < 1) {
            throw new ServiceException("扣减用户积分失败，扣减的积分分值不能小于0。");
        }

        this.transactionTemplate.executeWithoutResult(status -> {
			try {
				String userName = detail.getUserName();
				BbsUserPoints dbUserPoints = this.pointsServiceImpl.getAndInitUserPoints(userName);
				Integer beforeCurrentPoints = dbUserPoints.getCurrentPoints();
				if (points > beforeCurrentPoints) {
					throw new ServiceException("扣减用户积分失败，当前用户可用积分不足。");
				}

				// 更新用户积分
				int afterCurrentPoints = beforeCurrentPoints - points;
				BbsUserPoints updUserPoints = new BbsUserPoints();
				updUserPoints.setId(dbUserPoints.getId());
				updUserPoints.setCurrentPoints(afterCurrentPoints);
				this.updateById(updUserPoints);

				// 添加用户积分明细
				BbsUserPointsDetail newDetail = new BbsUserPointsDetail();
				newDetail.setUserName(userName);
				newDetail.setDetailName(detail.getDetailName());
				newDetail.setDetailCode(detail.getDetailCode());
				newDetail.setDetailPoints(points);
				newDetail.setBeforePoints(beforeCurrentPoints);
				newDetail.setAfterPoints(afterCurrentPoints);
				newDetail.setDescription(detail.getDescription());
				this.pointsDetailService.save(newDetail);

			} catch (Exception e) {
				status.setRollbackOnly();
				throw new ServiceException(e.getMessage());
			}
        });
    }

}
