package com.tangguo.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.utils.SecurityUtils;
import com.tangguo.domain.BbsUserPoints;
import com.tangguo.domain.BbsUserPointsDetail;
import com.tangguo.domain.dto.PointsDetailDTO;
import com.tangguo.mapper.BbsUserPointsMapper;
import com.tangguo.service.IBbsUserPointsDetailService;
import com.tangguo.service.IBbsUserPointsService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionManager;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * 用户积分Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Service
public class BbsUserPointsServiceImpl extends ServiceImpl<BbsUserPointsMapper, BbsUserPoints> implements IBbsUserPointsService {

    @Resource
    private BbsUserPointsMapper bbsUserPointsMapper;

    @Resource
    private IBbsUserPointsDetailService pointsDetailService;

    @Resource
    private BbsUserPointsServiceImpl userPointsServiceImpl;


    /**
     * 查询用户积分列表
     *
     * @param points 用户积分
     * @return 用户积分
     */
    @Override
    public List<BbsUserPoints> selectBbsUserPointsList(BbsUserPoints points) {
        return this.baseMapper.selectBbsUserPointsList(points);
    }


    /**
     * 查询用户积分
     *
     * @param userName 用户名
     * @return 积分
     */
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    @Override
    public BbsUserPoints getUserPoints(String userName) {
        BbsUserPoints dbUserPoints = this.getOne(
            Wrappers.lambdaQuery(BbsUserPoints.class).eq(BbsUserPoints::getUserName, userName)
        );
        if (Objects.isNull(dbUserPoints)) {
            dbUserPoints = new BbsUserPoints();
            dbUserPoints.setUserName(userName);
            dbUserPoints.setAccumulatedPoints(0);
            dbUserPoints.setCurrentPoints(0);
            this.save(dbUserPoints);
        }
        return dbUserPoints;
    }


    /**
     * 增加用户积分
     *
     * @param points 积分
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void addUserPoints(BbsUserPoints points) {
        String userName = points.getUserName();
        PointsDetailDTO detail = new PointsDetailDTO();
        detail.setDetailName("后台手动增加");
        detail.setDescription(String.format("管理员【%s】后台手动增加用户【%s】积分", SecurityUtils.getUsername(), userName));
        this.incrUserPoints(userName, points.getIncrOrDecrPoints(), detail);
    }


    /**
     * 扣减用户积分
     *
     * @param points 积分
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteUserPoints(BbsUserPoints points) {
        String userName = points.getUserName();
        PointsDetailDTO detail = new PointsDetailDTO();
        detail.setDetailName("后台手动扣减");
        detail.setDescription(String.format("管理员【%s】后台手动扣减用户【%s】积分", SecurityUtils.getUsername(), userName));
        this.decrUserPoints(userName, points.getIncrOrDecrPoints(), detail);
    }


    /**
     * 增加用户积分
     *
     * @param userName 用户名
     * @param points   积分
     * @param detail   积分明细
     */
    @Override
    public void incrUserPoints(String userName, int points, PointsDetailDTO detail) {
        synchronized (userName.intern()) {
            this.userPointsServiceImpl.execIncrUserPoints(userName, points, detail);
        }
    }


    /**
     * 扣减用户积分
     *
     * @param userName 用户名
     * @param points   积分
     * @param detail   积分明细
     */
    @Override
    public void decrUserPoints(String userName, int points, PointsDetailDTO detail) {
        synchronized (userName.intern()) {
            this.userPointsServiceImpl.execDecrUserPoints(userName, points, detail);
        }
    }


    /**
     * 增加用户积分
     *
     * @param userName 用户名
     * @param points   积分
     * @param detail   积分明细
     */
    @Transactional(rollbackFor = Exception.class)
    public void execIncrUserPoints(String userName, int points, PointsDetailDTO detail) {
        if (points < 1) {
            throw new ServiceException("增加用户积分失败，增加的积分分值不能小于0。");
        }

        // 更新用户积分
        BbsUserPoints userPoints = this.userPointsServiceImpl.getUserPoints(userName);
        Integer currentPoints = userPoints.getCurrentPoints();
        int incrCurrentPoints = currentPoints + points;
        userPoints.setCurrentPoints(incrCurrentPoints);
        userPoints.setAccumulatedPoints(userPoints.getAccumulatedPoints() + points);
        this.updateById(userPoints);

        // 添加用户积分明细
        BbsUserPointsDetail newDetail = new BbsUserPointsDetail();
        newDetail.setUserName(userName);
        newDetail.setDetailName(detail.getDetailName());
        newDetail.setDetailCode(detail.getDetailCode());
        newDetail.setDetailPoints(points);
        newDetail.setBeforePoints(currentPoints);
        newDetail.setAfterPoints(incrCurrentPoints);
        newDetail.setDescription(detail.getDescription());
        this.pointsDetailService.save(newDetail);
    }


    /**
     * 扣减用户积分
     *
     * @param userName 用户名
     * @param points   积分
     * @param detail   积分明细
     */
    @Transactional(rollbackFor = Exception.class)
    public void execDecrUserPoints(String userName, int points, PointsDetailDTO detail) {
        if (points < 1) {
            throw new ServiceException("扣减用户积分失败，扣减的积分分值不能小于0。");
        }

        // 更新用户积分
        BbsUserPoints userPoints = this.userPointsServiceImpl.getUserPoints(userName);
        Integer currentPoints = userPoints.getCurrentPoints();
        if (points > currentPoints) {
            throw new ServiceException("扣减用户积分失败，当前用户可用积分不足。");
        }

        int decrCurrentPoints = currentPoints - points;
        userPoints.setCurrentPoints(decrCurrentPoints);
        this.updateById(userPoints);

        // 添加用户积分明细
        BbsUserPointsDetail newDetail = new BbsUserPointsDetail();
        newDetail.setUserName(userName);
        newDetail.setDetailName(detail.getDetailName());
        newDetail.setDetailCode(detail.getDetailCode());
        newDetail.setDetailPoints(points);
        newDetail.setBeforePoints(currentPoints);
        newDetail.setAfterPoints(decrCurrentPoints);
        newDetail.setDescription(detail.getDescription());
        this.pointsDetailService.save(newDetail);
    }

}
