package com.tangguo.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.domain.BbsPointsGoods;
import com.tangguo.domain.BbsUserPoints;
import com.tangguo.domain.BbsUserPointsExchange;
import com.tangguo.domain.dto.PointsDetail;
import com.tangguo.mapper.BbsUserPointsExchangeMapper;
import com.tangguo.service.IBbsPointsGoodsService;
import com.tangguo.service.IBbsUserPointsExchangeService;
import com.tangguo.service.IBbsUserPointsService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * 用户积分兑换Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-29
 */
@Service
public class BbsUserPointsExchangeServiceImpl extends ServiceImpl<BbsUserPointsExchangeMapper, BbsUserPointsExchange> implements IBbsUserPointsExchangeService {

    @Resource
    private BbsUserPointsExchangeMapper bbsUserPointsExchangeMapper;

    @Resource
    private IBbsUserPointsService userPointsService;

    @Resource
    private IBbsPointsGoodsService pointsGoodsService;


    /**
     * 查询用户积分兑换列表
     *
     * @param exchange 用户积分兑换
     * @return 用户积分兑换
     */
    @Override
    public List<BbsUserPointsExchange> selectBbsUserPointsExchangeList(BbsUserPointsExchange exchange) {
	    return this.baseMapper.selectBbsUserPointsExchangeList(exchange);
    }


    /**
     * 添加用户积分兑换记录
     *
     * @param exchange 兑换
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void addUserPointsExchange(BbsUserPointsExchange exchange) {
        // 积分商品校验
        BbsPointsGoods dbGoods = this.pointsGoodsService.getByGoodsCode(exchange.getGoodsCode());
        if (Objects.isNull(dbGoods)) {
            throw new ServiceException("兑换失败，未查询到当前兑换的商品数据。");
        }
        if (dbGoods.getStockNum() < 1) {
            throw new ServiceException("兑换失败，当前兑换的商品库存数量不足。");
        }

        // 用户积分校验
        String userName = exchange.getUserName();
        Integer exchangePoints = dbGoods.getExchangePoints();
        BbsUserPoints dbUserPoints = this.userPointsService.getUserPoints(userName);
        int currentPoints = dbUserPoints.getCurrentPoints() - dbGoods.getExchangePoints();
        if (currentPoints < 0) {
            throw new ServiceException("兑换失败，当前用户可用积分不足。");
        }

        // 扣减商品库存
        Long goodsId = dbGoods.getId();
        boolean decrResult = this.pointsGoodsService.decrGoodsStock(goodsId, 1);
        if (!decrResult) {
            throw new ServiceException("兑换失败，当前兑换的商品库存数量不足。");
        }

        // 增加商品销量
        this.pointsGoodsService.incrGoodsSales(goodsId, 1);

        // 扣减用户积分
        PointsDetail detail = new PointsDetail();
        detail.setUserName(userName);
        detail.setDetailPoints(exchangePoints);
        detail.setDetailName(dbGoods.getName());
        detail.setDetailCode(dbGoods.getCode());
        detail.setDescription(String.format("用户【%s】兑换积分商品【%s-%s】", userName, dbGoods.getName(), dbGoods.getCode()));
        this.userPointsService.decrUserPoints(detail);

        // 添加兑换记录
        BbsUserPointsExchange addExchange = new BbsUserPointsExchange();
        addExchange.setNikeName(exchange.getNikeName());
        addExchange.setUserName(exchange.getUserName());
        addExchange.setUserGradeName(exchange.getUserGradeName());
        addExchange.setGoodsName(dbGoods.getName());
        addExchange.setGoodsCode(dbGoods.getCode());
        addExchange.setGoodsExchangePoints(dbGoods.getExchangePoints());
        addExchange.setUserRemainingPoints(currentPoints);
        this.save(addExchange);
    }

}
