package com.tangguo.controller.mobile;

import com.tangguo.common.core.domain.AjaxResult;
import com.tangguo.common.core.page.TableDataInfo;
import com.tangguo.common.mauth.MobileAuth;
import com.tangguo.common.mauth.MobileTokenHelper;
import com.tangguo.common.utils.PageUtils;
import com.tangguo.common.utils.ValidateOperations;
import com.tangguo.domain.bo.*;
import com.tangguo.domain.vo.BbsCommentDetailVO;
import com.tangguo.domain.vo.BbsMomentListVO;
import com.tangguo.domain.vo.BbsVoteOptionVO;
import com.tangguo.domain.vo.BbsVoteVO;
import com.tangguo.service.IBbsMomentCommentService;
import com.tangguo.service.IBbsMomentService;
import com.tangguo.service.IBbsMomentVoteOptionService;
import com.tangguo.service.IBbsMomentVoteService;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 移动端动态控制器
 *
 * @author 谈笑
 * @createTime 2025-09-02 16:19:17 星期二
 */
@RestController
@RequestMapping("/bbs/mobile/moment")
public class MBbsMomentController {

	@Resource
	private IBbsMomentService momentService;

	@Resource
	private IBbsMomentCommentService momentCommentService;

	@Resource
	private IBbsMomentVoteService momentVoteService;

	@Resource
	private IBbsMomentVoteOptionService momentVoteOptionService;


	/**
	 * 查询动态列表
	 *
	 * @return 动态
	 */
	@MobileAuth
	@GetMapping("/moments")
	public TableDataInfo getMoments(QueryMomentBO bo) {
		PageUtils.clearPage();
		List<BbsMomentListVO> moments = this.momentService.getMoments(bo);
		return TableDataInfo.getDataTable(moments);
	}


	/**
	 * 查询动态详情
	 *
	 * @return 动态
	 */
	@MobileAuth
	@GetMapping("/{id}")
	public AjaxResult getMomentDetail(@PathVariable Long id) {
		BbsMomentListVO dbMoment = this.momentService.getMomentDetail(id);
		return AjaxResult.success(dbMoment);
	}


	/**
	 * 创建动态
	 *
	 * @param bo 动态
	 * @return 创建结果
	 */
	@MobileAuth
	@PostMapping("/create")
	public AjaxResult createMoment(@RequestBody CreateMomentBO bo) {
		this.momentService.userCreateMoment(bo);
		return AjaxResult.success();
	}


	/**
	 * 删除动态
	 *
	 * @param momentId 动态Id
	 * @return 删除结果
	 */
	@MobileAuth
	@DeleteMapping("/delete/{momentId}")
	public AjaxResult deleteMoment(@PathVariable Long momentId) {
		this.momentService.userDeleteMoment(momentId);
		return AjaxResult.success();
	}


	/**
	 * 点赞动态
	 *
	 * @param bo 动态
	 * @return 点赞结果
	 */
	@MobileAuth
	@PostMapping("/like")
	public AjaxResult likeMoment(@RequestBody LikeMomentBO bo) {
		ValidateOperations.generalValidate(bo);
		synchronized (String.valueOf(bo.getMomentId()).intern()) {
			this.momentService.userLikeMoment(bo);
			return AjaxResult.success();
		}
	}


	/**
	 * 投票动态
	 *
	 * @param bo 动态
	 * @return 投票结果
	 */
	@MobileAuth
	@PostMapping("/vote")
	public AjaxResult voteMoment(@RequestBody VoteMomentBO bo) {
		ValidateOperations.generalValidate(bo);
		synchronized (String.valueOf(bo.getMomentId()).intern()) {
			this.momentService.userVoteMoment(bo);
			return AjaxResult.success();
		}
	}


	/**
	 * 投票动态统计
	 *
	 * @param momentId 动态Id
	 * @return 投票结果
	 */
	@MobileAuth
	@GetMapping("/vote/count")
	public AjaxResult getMomentVoteCount(@RequestParam Long momentId) {
		List<BbsVoteOptionVO> dbOptionCount = this.momentVoteOptionService.getVoteOptionCount(momentId);
		int voteCount = dbOptionCount.stream().mapToInt(BbsVoteOptionVO::getVoteCount).sum();
		Map<String, Object> voteOptionMap = new LinkedHashMap<>(2);
		voteOptionMap.put("voteOptionCount", dbOptionCount);
		voteOptionMap.put("voteCount", voteCount);
		return AjaxResult.success(voteOptionMap);
	}


	/**
	 * 动态投票记录
	 *
	 * @param momentId 动态Id
	 * @return 投票记录
	 */
	@MobileAuth
	@GetMapping("/votes")
	public TableDataInfo getMomentVotes(@RequestParam Long momentId, @RequestParam String optionCode) {
		PageUtils.startPage();
		List<BbsVoteVO> dbVotes = this.momentVoteService.getMomentVotes(momentId, optionCode);
		return TableDataInfo.getDataTable(dbVotes);
	}


	/**
	 * 查询动态评论
	 *
	 * @return 动态
	 */
	@MobileAuth
	@GetMapping("/comments")
	public TableDataInfo getMomentComments(@RequestParam Long momentId) {
		String userName = MobileTokenHelper.getUserName();
		PageUtils.clearPage();
		List<BbsCommentDetailVO> dbComments = this.momentCommentService.getMomentComments(momentId, userName);
		return TableDataInfo.getDataTable(dbComments);
	}


	/**
	 * 评论动态
	 *
	 * @param bo 动态
	 * @return 投票结果
	 */
	@MobileAuth
	@PostMapping("/comment")
	public AjaxResult commentMoment(@RequestBody CommentMomentBO bo) {
		ValidateOperations.generalValidate(bo);
		synchronized (String.valueOf(bo.getMomentId()).intern()) {
			this.momentService.userCommentMoment(bo);
			return AjaxResult.success();
		}
	}


	/**
	 * 删除评论
	 *
	 * @param commentId 评论Id
	 * @return 删除结果
	 */
	@MobileAuth
	@DeleteMapping("/comment/delete/{commentId}")
	public AjaxResult deleteComment(@PathVariable Long commentId) {
		this.momentService.userDeleteComment(commentId);
		return AjaxResult.success();
	}


	/**
	 * 精选评论
	 *
	 * @param bo 评论
	 * @return 精选结果
	 */
	@MobileAuth
	@PostMapping("/comment/featured")
	public AjaxResult featuredComment(@RequestBody FeaturedCommentBO bo) {
		ValidateOperations.generalValidate(bo);
		this.momentService.userFeaturedComment(bo);
		return AjaxResult.success();
	}

}
