package com.tangguo.controller.pc;

import com.tangguo.common.annotation.Log;
import com.tangguo.common.core.controller.BaseController;
import com.tangguo.common.core.domain.AjaxResult;
import com.tangguo.common.core.page.TableDataInfo;
import com.tangguo.common.enums.BusinessType;
import com.tangguo.domain.BbsSensitiveWord;
import com.tangguo.domain.vo.DataImportResult;
import com.tangguo.service.IBbsSensitiveWordService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.util.List;

/**
 * 敏感词库Controller
 *
 * @author ruoyi
 * @date 2025-08-27
 */
@Api(tags = "敏感词库管理")
@RestController
@RequestMapping("/bbs/moment/sensitive/word")
public class BbsSensitiveWordController extends BaseController {

	@Resource
	private IBbsSensitiveWordService bbsSensitiveWordService;


	/**
	 * 查询敏感词库列表
	 */
	@ApiOperation("查询敏感词库列表")
	@PreAuthorize("@ss.hasPermi('bbs:word:list')")
	@GetMapping("/list")
	public TableDataInfo list(BbsSensitiveWord bbsSensitiveWord) {
		startPage();
		List<BbsSensitiveWord> list = bbsSensitiveWordService.selectBbsSensitiveWordList(bbsSensitiveWord);
		return getDataTable(list);
	}


	/**
	 * 获取敏感词库详细信息
	 */
	@ApiOperation("获取敏感词库详细信息")
	@PreAuthorize("@ss.hasPermi('bbs:word:query')")
	@GetMapping(value = "/{id}")
	public AjaxResult getInfo(@PathVariable("id") Long id) {
		return success(bbsSensitiveWordService.getById(id));
	}


	/**
	 * 新增敏感词库
	 */
	@ApiOperation("新增敏感词库")
	@PreAuthorize("@ss.hasPermi('bbs:word:add')")
	@Log(title = "敏感词库", businessType = BusinessType.INSERT)
	@PostMapping
	public AjaxResult add(@RequestBody BbsSensitiveWord word) {
		this.bbsSensitiveWordService.addWord(word);
		return AjaxResult.success();
	}


	/**
	 * 修改敏感词库
	 */
	@ApiOperation("修改敏感词库")
	@PreAuthorize("@ss.hasPermi('bbs:word:edit')")
	@Log(title = "敏感词库", businessType = BusinessType.UPDATE)
	@PutMapping
	public AjaxResult edit(@RequestBody BbsSensitiveWord word) {
		this.bbsSensitiveWordService.editWord(word);
		return AjaxResult.success();
	}


	/**
	 * 删除敏感词库
	 */
	@ApiOperation("删除敏感词库")
	@PreAuthorize("@ss.hasPermi('bbs:word:remove')")
	@Log(title = "敏感词库", businessType = BusinessType.DELETE)
	@DeleteMapping("/{id}")
	public AjaxResult remove(@PathVariable Long id) {
		this.bbsSensitiveWordService.deleteWord(id);
		return AjaxResult.success();
	}


	/**
	 * 导入敏感词库
	 */
	@ApiOperation("导入敏感词库")
	@PreAuthorize("@ss.hasPermi('bbs:word:import')")
	@Log(title = "敏感词库", businessType = BusinessType.IMPORT)
	@PostMapping("/import")
	public AjaxResult importWords(MultipartFile file) {
		DataImportResult<String> result = this.bbsSensitiveWordService.importWords(file);
		return AjaxResult.success(result);
	}


	/**
	 * 刷新敏感词库缓存
	 */
	@ApiOperation("刷新敏感词库缓存")
	@PreAuthorize("@ss.hasPermi('bbs:word:refresh')")
	@Log(title = "敏感词库", businessType = BusinessType.UPDATE)
	@PutMapping("/refresh")
	public AjaxResult refresh() {
		this.bbsSensitiveWordService.reloadCacheWords();
		return AjaxResult.success();
	}

}
