package com.tangguo.service.impl;

import cn.hutool.core.util.IdUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.utils.StringUtils;
import com.tangguo.domain.BbsPointsGoods;
import com.tangguo.mapper.BbsPointsGoodsMapper;
import com.tangguo.service.IBbsPointsGoodsService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * 积分商品Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-28
 */
@Service
public class BbsPointsGoodsServiceImpl extends ServiceImpl<BbsPointsGoodsMapper, BbsPointsGoods> implements IBbsPointsGoodsService {

	@Resource
	private BbsPointsGoodsMapper bbsPointsGoodsMapper;


	/**
	 * 查询积分商品列表
	 *
	 * @param goods 积分商品
	 * @return 积分商品
	 */
	@Override
	public List<BbsPointsGoods> selectBbsPointsGoodsList(BbsPointsGoods goods) {
		LambdaQueryWrapper<BbsPointsGoods> lqw = Wrappers.lambdaQuery();
		lqw.orderByDesc(BbsPointsGoods::getCreateTime);
		lqw.like(StringUtils.isNotBlank(goods.getName()), BbsPointsGoods::getName, goods.getName());
		lqw.eq(StringUtils.isNotBlank(goods.getCode()), BbsPointsGoods::getCode, goods.getCode());
		lqw.eq(StringUtils.isNotBlank(goods.getImgsUrl()), BbsPointsGoods::getImgsUrl, goods.getImgsUrl());
		lqw.eq(goods.getSalesPrice() != null, BbsPointsGoods::getSalesPrice, goods.getSalesPrice());
		lqw.eq(goods.getExchangePoints() != null, BbsPointsGoods::getExchangePoints, goods.getExchangePoints());
		lqw.eq(goods.getStockNum() != null, BbsPointsGoods::getStockNum, goods.getStockNum());
		lqw.eq(goods.getSalesNum() != null, BbsPointsGoods::getSalesNum, goods.getSalesNum());
		lqw.eq(goods.getShowStatus() != null, BbsPointsGoods::getShowStatus, goods.getShowStatus());
		return bbsPointsGoodsMapper.selectList(lqw);
	}


	/**
	 * 添加积分商品
	 *
	 * @param goods 积分商品
	 */
	@Override
	public void addGoods(BbsPointsGoods goods) {
		long nameCount = this.count(
			Wrappers.lambdaQuery(BbsPointsGoods.class).eq(BbsPointsGoods::getName, goods.getName())
		);
		if (nameCount > 0) {
			throw new ServiceException("添加失败，当前商品名称已存在。");
		}

		BbsPointsGoods addGoods = new BbsPointsGoods();
		addGoods.setName(goods.getName());
		addGoods.setCode(IdUtil.fastSimpleUUID());
		addGoods.setImgsUrl(goods.getImgsUrl());
		addGoods.setSalesPrice(goods.getSalesPrice());
		addGoods.setExchangePoints(goods.getExchangePoints());
		addGoods.setStockNum(goods.getStockNum());
		addGoods.setSalesNum(0);
		addGoods.setShowStatus(goods.getShowStatus());
		this.save(addGoods);
	}


	/**
	 * 修改积分商品
	 *
	 * @param goods 积分商品
	 */
	@Override
	public void editGoods(BbsPointsGoods goods) {
		BbsPointsGoods dbGoods = this.getById(goods.getId());
		if (Objects.isNull(dbGoods)) {
			throw new ServiceException("修改失败，未查询到当前商品数据。");
		}

		if (!dbGoods.getName().equals(goods.getName())) {
			long nameCount = this.count(
				Wrappers.lambdaQuery(BbsPointsGoods.class).eq(BbsPointsGoods::getName, goods.getName())
			);
			if (nameCount > 0) {
				throw new ServiceException("修改失败，当前商品名称已存在。");
			}
		}

		BbsPointsGoods addGoods = new BbsPointsGoods();
		addGoods.setId(dbGoods.getId());
		addGoods.setName(goods.getName());
		addGoods.setImgsUrl(goods.getImgsUrl());
		addGoods.setSalesPrice(goods.getSalesPrice());
		addGoods.setExchangePoints(goods.getExchangePoints());
		addGoods.setStockNum(goods.getStockNum());
		addGoods.setShowStatus(goods.getShowStatus());
		this.updateById(addGoods);
	}


	/**
	 * 删除积分商品
	 *
	 * @param goodsId 积分商品Id
	 */
	@Override
	public void deleteGoods(Long goodsId) {
		this.removeById(goodsId);
	}

}
