package com.tangguo.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.common.exception.ServiceException;
import com.tangguo.common.utils.StringUtils;
import com.tangguo.domain.BbsTopic;
import com.tangguo.enums.IsTop;
import com.tangguo.mapper.BbsTopicMapper;
import com.tangguo.service.IBbsTopicService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 话题Service业务层处理
 *
 * @author ruoyi
 * @date 2025-08-27
 */
@Service
public class BbsTopicServiceImpl extends ServiceImpl<BbsTopicMapper, BbsTopic> implements IBbsTopicService {

	@Resource
	private BbsTopicMapper bbsTopicMapper;


	/**
	 * 查询话题列表
	 *
	 * @param topic 话题
	 * @return 话题
	 */
	@Override
	public List<BbsTopic> selectBbsTopicList(BbsTopic topic) {
		LambdaQueryWrapper<BbsTopic> lqw = Wrappers.lambdaQuery();
		lqw.orderByDesc(BbsTopic::getCreateTime);
		lqw.like(StringUtils.isNotBlank(topic.getSource()), BbsTopic::getSource, topic.getSource());
		lqw.like(StringUtils.isNotBlank(topic.getName()), BbsTopic::getName, topic.getName());
		lqw.orderByAsc(BbsTopic::getIsTop);
		lqw.orderByDesc(BbsTopic::getTopTime);
		lqw.orderByAsc(BbsTopic::getSort);
		return bbsTopicMapper.selectList(lqw);
	}


	/**
	 * 新增话题
	 *
	 * @param topic 话题
	 */
	@Override
	public void addTopic(BbsTopic topic) {
		long nameCount = this.count(
			Wrappers.lambdaQuery(BbsTopic.class).eq(BbsTopic::getName, topic.getName())
		);
		if (nameCount > 0) {
			throw new ServiceException("新增失败，当前话题名称已存在。");
		}

		BbsTopic newTopic = new BbsTopic();
		newTopic.setSource("后台管理");
		newTopic.setName(topic.getName());
		newTopic.setSort(topic.getSort());
		this.save(newTopic);
	}


	/**
	 * 修改话题
	 *
	 * @param topic 话题
	 */
	@Override
	public void editeTopic(BbsTopic topic) {
		BbsTopic dbTopic = this.getById(topic.getId());
		if (Objects.isNull(dbTopic)) {
			throw new ServiceException("修改失败，当前话题数据不存在。");
		}

		if (!dbTopic.getName().equals(topic.getName())) {
			long nameCount = this.count(
				Wrappers.lambdaQuery(BbsTopic.class).eq(BbsTopic::getName, topic.getName())
			);
			if (nameCount > 0) {
				throw new ServiceException("修改失败，当前话题名称已存在。");
			}
		}

		BbsTopic updTopic = new BbsTopic();
		updTopic.setId(dbTopic.getId());
		updTopic.setName(topic.getName());
		updTopic.setSort(topic.getSort());
		this.updateById(updTopic);
	}


	/**
	 * 置顶话题
	 *
	 * @param topic 话题
	 */
	@Override
	public void topTopic(BbsTopic topic) {
		BbsTopic dbTopic = this.getById(topic.getId());
		if (Objects.isNull(dbTopic)) {
			throw new ServiceException("置顶失败，当前话题数据不存在。");
		}

		Integer isTop = topic.getIsTop();
		if (!dbTopic.getIsTop().equals(isTop)) {
			LambdaUpdateWrapper<BbsTopic> wrapper = Wrappers.lambdaUpdate(BbsTopic.class);
			wrapper.eq(BbsTopic::getId, dbTopic.getId());
			if (IsTop.TOP.getIsTop() == isTop) {
				wrapper.set(BbsTopic::getIsTop, IsTop.TOP.getIsTop());
				wrapper.set(BbsTopic::getTopTime, new Date());
			} else {
				wrapper.set(BbsTopic::getIsTop, IsTop.NOT_TOP.getIsTop());
				wrapper.set(BbsTopic::getTopTime, null);
			}
			this.update(wrapper);
		}
 	}


	/**
	 * 删除话题
	 *
	 * @param topicId 话题Id
	 */
	@Override
	public void deleteTopic(Long topicId) {
		this.removeById(topicId);
	}

}
