package com.tangguo.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tangguo.domain.BbsMoment;
import com.tangguo.domain.BbsMomentVoteOption;
import com.tangguo.domain.bo.CreateMomentBO;
import com.tangguo.enums.VoteOptionType;
import com.tangguo.mapper.BbsMomentVoteOptionMapper;
import com.tangguo.service.IBbsMomentVoteOptionService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * 动态投票选项Service业务层处理
 *
 * @author ruoyi
 * @date 2025-09-01
 */
@Service
public class BbsMomentVoteOptionServiceImpl extends ServiceImpl<BbsMomentVoteOptionMapper, BbsMomentVoteOption> implements IBbsMomentVoteOptionService {

    @Resource
    private BbsMomentVoteOptionMapper bbsMomentVoteOptionMapper;


    /**
     * 查询动态投票选项
     *
     * @param momentId 动态Id
     * @return 投票选项
     */
    @Override
    public List<BbsMomentVoteOption> getVoteOptions(Long momentId) {
        return this.list(
            Wrappers.lambdaQuery(BbsMomentVoteOption.class)
                .eq(BbsMomentVoteOption::getMomentId, momentId)
        );
    }


    /**
     * 查询动态投票选项
     *
     * @param momentIds 动态Ids
     * @return 投票选项
     */
    @Override
    public List<BbsMomentVoteOption> getVoteOptions(List<Long> momentIds) {
        List<BbsMomentVoteOption> dbOptions = new ArrayList<>(2);
        if (CollUtil.isNotEmpty(momentIds)) {
            dbOptions = this.list(
                Wrappers.lambdaQuery(BbsMomentVoteOption.class).in(BbsMomentVoteOption::getMomentId, momentIds)
            );
        }
        return dbOptions;
    }


    /**
     * 添加动态投票选项
     *
     * @param moment 动态
     * @param voteOptions 投票选项
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void addVoteOptions(BbsMoment moment, List<CreateMomentBO.VoteOption> voteOptions) {
        if (CollUtil.isNotEmpty(voteOptions)) {
            List<BbsMomentVoteOption> newOptions = new ArrayList<>(voteOptions.size());
            for (int i = 0; i < voteOptions.size(); i++) {
                CreateMomentBO.VoteOption option = voteOptions.get(i);
                BbsMomentVoteOption newOption = new BbsMomentVoteOption();
                newOption.setMomentId(moment.getId());
                VoteOptionType optionType = VoteOptionType.getVoteOptionType(moment.getVoteOptionType());
                newOption.setType(optionType.name());
                newOption.setName(option.getName());
                int optionIndex = i + 1;
                newOption.setCode(String.valueOf(optionIndex));
                newOption.setSort(optionIndex);
                if (VoteOptionType.IMAGE == optionType) {
                    newOption.setImageUrl(option.getImageUrl());
                }
                newOptions.add(newOption);
            }
            this.saveBatch(newOptions);
        }
    }


    /**
     * 删除动态投票选项
     *
     * @param momentId 动态Id
     */
    @Override
    public void deleteVoteOptions(Long momentId) {
        this.remove(
            Wrappers.lambdaQuery(BbsMomentVoteOption.class).eq(BbsMomentVoteOption::getMomentId, momentId)
        );
    }


    /**
     * 查询投票选项
     *
     * @param momentId   动态Id
     * @param optionCode 投票选项编码
     * @return 投票选项
     */
    @Override
    public BbsMomentVoteOption getVoteOption(Long momentId, String optionCode) {
        return this.getOne(
            Wrappers.lambdaQuery(BbsMomentVoteOption.class)
                .eq(BbsMomentVoteOption::getMomentId, momentId)
                .eq(BbsMomentVoteOption::getCode, optionCode)
        );
    }

}
